/*********************************************************************/
/*                                                                   */
/*  MACRO: OMABAKUP 913SP4  20Feb2009                                */
/*         Add dynamic encoding of XML formatting                    */
/*                                                                   */
/*  USAGE:                                                           */
/*                                                                   */
/* options metaserver='localhost'                                    */
/*         metaport=8561                                             */
/*         metaprotocol=bridge                                       */
/*         metauser='domain\userid'                                  */
/*         metapass='{sas001}eW91clBhc3N3b3JkSGVyZQ=='               */
/*         metarepository='Foundation';                              */
/* %OMABAKUP( DestinationPath="C:\SAS\9.1\Lev1\SASBackup",           */
/*            ServerStartPath="C:\SAS\9.1\Lev1\SASMain",             */
/*            RposmgrPath="MetadataServer\rposmgr",                  */
/*            Reorg="Y", Restore="N", RunAnalysis="N",               */
/*            RepositoryList="Foundation,Project One")               */
/*                                                                   */
/*  DESCRIPTION:                                                     */
/*    This macro will backup one or all repositories on a server.    */
/*    If the REORG parameter is set to Y, will reorganize the data   */
/*    in the repository data sets to reclaim disk space for deleted  */
/*    metadata objects. RESTORE copies files from the backup location*/
/*    to the registered repository location. The RUNANALYSIS option  */
/*    runs a variable analysis job (omavaran macro) against the      */
/*    backup copy of each repository to optimize memory usage.       */
/*    REPOSITORYLIST allows the user to specify a subset of one or   */
/*    more registered repositories to backup or restore.             */
/*                                                                   */
/*  PARAMETERS:                                                      */
/*     DestinationPath: The absolute pathname of the directory where */
/*         the backup copy is to be stored.  This should be a        */
/*         directory that already exists.                            */
/*         Default: "C:\SAS\9.1\Lev1\SASBackup"                      */
/*     ServerStartPath: The absolute pathname of the directory in    */
/*         which the metadata server was started.                    */
/*         Default: "C:\SAS\9.1\Lev1\SASMain"                        */
/*     RposmgrPath: The relative pathname of the repository manager  */
/*         as defined in the omaconfig.xml file.                     */
/*         Default: "MetadataServer\rposmgr"                         */
/*     Reorg: Use proc copy to reclaim space for deleted records and */
/*         then OS copy the Reorg'd files in the destination         */
/*         directory back to the relative repository directories.    */
/*         Default: Reorg="N":                                       */
/*         Restore="Y" turns this option off.                        */
/*     Restore: Restore repository files from the backup location to */
/*         the original repository location.                         */
/*         Default: Restore="N"                                      */
/*     RepositoryList: List of one or more repositories to process.  */
/*         Supports the keywords ALL and REPOSMGR                    */
/*         Default: RepositoryList="ALL"(all registered repositories)*/
/*     RunAnalysis: Y => After repositories are back online, runs the*/
/*         variable analysis job against each backup copy and stores */
/*         the results in the DS_VCHAR file in the backup location.  */
/*         The next backup job will merge the DS_VCHAR results into  */
/*         the DS_DEFN file located in the registered repository     */
/*         location and deletes the DS_VCHAR from the backup location*/
/*         Default: RunAnalysis="N"                                  */
/*         Restore="Y" turns this option off.                        */
/*                                                                   */
/*                                                                   */
/*  NOTES:                                                           */
/*    This macro runs properly on Windows, UNIX, Z/OS and NTSV hosts.*/
/*    For SP4 uses proc copy to copy repository data sets on all     */
/*    supported hosts.                                               */
/*                                                                   */
/*    Optionally macro can be modified to do X commands by changing  */
/*    macro variable USE_OS_COPIES=0 to USE_OS_COPIES=1. Search for  */
/*    USE_OS_COPIES assignment below. Note X commands are not        */
/*    permitted when submitted from a workspace server.              */
/*                                                                   */
/*    You may wish to not present your password in it's cleartext    */
/*    form in the source of a program.  The alternative is to use    */
/*    proc pwencode to encode it to a form that cannot be used to    */
/*    login. The result can be used in the METAPASS option.          */
/*    In any case you should guard your password carefully using     */
/*    private file permissions.                                      */
/*    proc pwencode in="yourPasswordHere"; run;                      */
/*                                                                   */
/*    SP3: Add support for WIN_NTSV; no DS_DEFN copy to REPOSMGR.    */
/*         Add Reorg option.                                         */
/*    SP4: Add Restore, RepositoryList & RunAnalysis functionality.  */
/*                                                                   */
/*********************************************************************/
/* You might need to edit this macro according to site specific needs. */
%macro OMABUCPY(repositoryName,registeredName,repositoryId,srcReposPath,
        DestinationPath,destPath,addRepositoryName=Y,Reorg=Y,Restore=N);
%global TYPE;

%let srcDirPath=%superq(srcReposPath);
%let DestinationPath=%superq(DestinationPath);
%let destPath=%superq(destPath);
%let repositoryName=%superq(repositoryName);
%let registeredName=%superq(registeredName);

/*
%put DEBUG: OMABUCPY: Id=&repositoryId;
%put DEBUG: OMABUCPY: RepositoryName=&repositoryName;
%put DEBUG: OMABUCPY: RegisteredName=&registeredName;
%put DEBUG: OMABUCPY: SourcePath=&srcDirPath;
%put DEBUG: OMABUCPY: DestinationPath=&DestinationPath;
%put DEBUG: OMABUCPY: DestPath=&destPath;
%put DEBUG: OMABUCPY: Restore=&Restore;
%put DEBUG: OMABUCPY: Reorg=&Reorg;
%put DEBUG: OMABUCPY: AddRepositoryName=&addRepositoryName;
*/

/* Set appropriate OS dependent values for srcDirPath and destDirPath. */
/* Create destination directories if they do not exist. */
%let createdDestDir=0;
%let destdir=%str();
%if &sysscp=WIN OR &sysscp=DNTHOST %then
  %do;
   %let tpath=&DestinationPath\&repositoryName;
   %if &addRepositoryName=Y %then
     %do;
       /* Create the destination directory if it does not exist */
       %if (%sysfunc(fileexist("&tpath")) eq 0) %then
        %do;
           %put NOTE: Creating destination directory &tpath;
           data _null_;
            destdir=dcreate("&repositoryName","&DestinationPath\");
            if destdir = ' ' then
             do;
               file print;
               put "ERROR: Could not &TYPE repository &registeredName";
               put 'See log for details';
               file log;
             end;
            call symputx("destdir",destdir);
          run;
          %put;
          %if %superq(destdir)^=%str() %then
             %let createdDestDir=1;
          %else
           %do;
             %put "WARNING: Could not create destination folder &tpath";
             %put ERROR: Could not &TYPE repository "&registeredName";
             %goto nocopy;
           %end;
        %end;
      %let destDirPath=&DestinationPath\&repositoryName;
     %end;
   %else
     %do;
       %let destDirPath=&DestinationPath;
     %end;
  %end;
%else  /* assume Z/OS, UNIX or UNIX compatible */
 %do;
   %let tpath=&DestinationPath/&repositoryName;
   %if &addRepositoryName=Y %then
     %do;
       /* Create the destination directory if it does not exist */
       %let tpath=&DestinationPath/&repositoryName;
       %if (%sysfunc(fileexist("&tpath")) eq 0) %then
        %do;
           %put NOTE: Creating destination directory &tpath;
           data _null_;
            destdir=dcreate("&repositoryName","&DestinationPath/");
            if destdir = ' ' then
             do;
               file print;
               put "ERROR: Could not &TYPE repository &registeredName";
               put 'See log for details';
               file log;
             end;
            call symputx("destdir",destdir);
           run;
           %put;
           %if %superq(destdir)^=%str() %then
             %let createdDestDir=1;
           %else
            %do;
             %put WARNING: Could not create destination folder "&tpath";
             %put ERROR: Could not &TYPE repository "&registeredName";
             %goto nocopy;
            %end;
        %end;
      %let destDirPath=&DestinationPath/&repositoryName;
     %end;
   %else
     %do;
        %let destDirPath=&DestinationPath;
     %end;
 %end;

%let destDirPath=%superq(destDirPath);
%let repositoryName=%superq(registeredName);

/*
%put;
%put DEBUG: NOTE: &TYPE Source Path is "&srcDirPath";
%put DEBUG: NOTE: &TYPE Destination Path is "&destDirPath";
%put;
%put DEBUG: nocopy; %goto nocopy;
*/


%if &Restore=N %then
 %do;
  /* append statements for this repository to var analysis job */
  data _null_;
    file OMAVARAN;

    /* put 'title Begin Analysis Of Repository "'"&repositoryName"'";'; */
    put "title Begin Analysis Of Repository &repositoryName;";
    put 'footnote SAS Metadata Server Backup Utility;';
    put 'options nocenter linesize=MAX;';
    put 'data _null_;';
    put 'file print;';
    put 'put ".";';
    put 'file log;';
    put 'run;';
    put '%put;';
    put '%put ***************************************************************;';
    put '%put              BEGIN ANALYSIS OF "'"&repositoryName"'";';
    put '%put ***************************************************************;';
    put '%put;';
    put '%BACKLIB('"&repositoryName"');';
    put '%omavaran (xyz, BACKLIB);';
    put "title End Analysis Of Repository &repositoryName;";
    put 'data _null_;';
    put 'file print;';
    put 'put ".";';
    put 'file log;';
    put 'run;';
    put '%put;';
    put '%put ***************************************************************;';
    put '%put              END ANALYSIS OF "'"&repositoryName"'";';
    put '%put ***************************************************************;';
    put '%put;';
  run;
 %end;

%let copyerr=0;

/* Insure we have a DS_DEFN data set in the repository so we can collect index
   statistics. */

%let srcliberr=0;
%let srcassgn=0;
%let destliberr=0;
%if &Restore=N %then
 %do;
   %put;
   %put NOTE: Create or update DS_DEFN in repository directory.;
   %put;
   libname src "%unquote(&srcDirPath)";
   %if &syslibrc ne 0 %then
     %do;
       %put WARNING: Could not assign libname to "&srcDirPath";
       %let srcliberr=1;
       %goto liberr;
     %end;
   %if (%sysfunc(exist(src.DS_DEFN))=0) %then
    %do;
      %let srcassgn=1;
      %let maxid=0;
      data src.DS_DEFN;
       length id $4 ds_name var_name $32 never_index var_chars lots_of_dups $1
       idx_used avg_idx_sel idx_rjct avg_idx_rjct_sel 8;
       ds_name='MDASSOC';
       never_index='Y';
       var_chars='';
       lots_of_dups='';
       idx_used = 0;
       avg_idx_sel = 0;
       idx_rjct = 0;
       avg_idx_rjct_sel = 0;
       id = put(1,pib4.);
       var_name='ASCMGRID';
       output;
       id = put(2,pib4.);
       var_name='PREDROLE';
       output;
       id = put(3,pib4.);
       var_name='SUCCROLE';
       output;
      stop;
      run;
    %end;

   /* Update DS_DEFN data set with any DS_VCHAR data from destination and index
      statistics -- exclude REPOSMGR */
   %if %superq(repositoryName)=REPOSMGR %then
       %put NOTE: DS_DEFN not updated in REPOSMGR repository directory.;
   %else
     %do;
       %*---------------------------------------------;
       %* Get max ID count from DS_DEFN      ;
       %*---------------------------------------------;
       %let maxid=0;
       proc sql NOPRINT;
          select put(max(input(id,pib4.)),10.) into :maxid from src.DS_DEFN;
          quit;

       %*---------------------------------------------------;
       %* If DS_VCHAR exists in dest then merge it          ;
       %* into the DS_DEFN dataset.  Only merge in the      ;
       %* attributes: id, ds_name, var_name, var_chars and  ;
       %* lots_of_dups from DS_VCHAR. Delete DS_VCHAR so it ;
       %* does not get copied back to the real repository   ;
       %* in a Reorg.                                       ;
       %*---------------------------------------------------;
       libname dest "%unquote(&destDirPath)";
       %if &syslibrc eq 0 %then
        %do;
          %if %sysfunc(exist(dest.DS_VCHAR)) %then
           %do;
             proc sort data=src.DS_DEFN; by ds_name var_name;run;
             data src.DS_DEFN;
               update src.DS_DEFN
               dest.DS_VCHAR(keep=id ds_name var_name var_chars lots_of_dups);
               by ds_name var_name;
               drop mid;
               if id = put(0,pib4.) then do;
                 mid = symget('maxid') + 1;
                 call symput('maxid',put(mid,10.));
                 id = put(mid,pib4.);
               end;
             run;
             proc datasets lib=dest nodetails;
               delete DS_VCHAR;
             quit;
           %end;
          %*---------------------------------------------;
          %* Update never_index field in DS_DEFN dataset ;
          %*---------------------------------------------;
          data src.DS_DEFN; set src.DS_DEFN;
            if idx_used = 0 and idx_rjct > 10000 then never_index='Y';
            if var_name='CLASNAME' then never_index='Y';
            if var_name='LDESC' then never_index='Y';
            if var_name='PREDROLE' then never_index='Y';
            if var_name='SUCCROLE' then never_index='Y';
            if var_name='ASCMGRID' then never_index='Y';
          run;
          libname dest clear;
        %end;
       %else
         %do;
          %let destliberr=1;
          %put WARNING: Could not assign libname to "&destDirPath";
         %end;
     %end;
  %if &srcassgn=1 %then
   %do;
     libname src clear;
   %end;
 %end;
%else
 %do;
   /* Restore=Y: Delete the DS_VCHAR file from this repository */
   /* if it exists in the backup location. */
   libname src "%unquote(&srcDirPath)";
   %if &syslibrc ne 0 %then
     %do;
       data _null_;
         file print;
         put "WARNING: Could not remove DS_VCHAR from &srcDirPath";
         put 'See log for details';
         file log;
       run;
       %put WARNING: Could not assign libname to "&srcDirPath";
       %put WARNING: Could not remove DS_VCHAR from "&srcDirPath";
     %end;
   %else
     %do;
       %if %sysfunc(exist(src.DS_VCHAR)) %then
        %do;
          %put NOTE: Deleting unused DS_VCHAR from "&srcDirPath";
          proc datasets lib=src nodetails;
            delete DS_VCHAR;
          quit;
        %end;
       libname src clear;
     %end;
 %end;

%liberr:
%put;
%put;
%if &srcliberr=1 or &destliberr=1 %then
 %do;
   data _null_;
     file print;
     put "ERROR: Could not &TYPE repository &repositoryName";
     put 'See log for details';
     file log;
   run;
   %put WARNING: Prior error prevented &TYPE of repository.;
   %put ERROR: Could not &TYPE repository "&repositoryName";
   %goto nocopy;
 %end;
%else %if &Restore=Y and &haveREPOSMGR eq 1 %then
 %do;
   /* Because this is a Restore the REPOSMGR might not have been included. */
   /* So, delete the DS_VCHAR in the REPOSMGR backup location it exists. */
   %if &sysscp=WIN OR &sysscp=DNTHOST %then
     %let path=&destPath\REPOSMGR;
   %else
     %let path=&destPath/REPOSMGR;
   libname reposmgr "%unquote(&path)";
   %if &syslibrc ne 0 %then
     %do;
       data _null_;
         file print;
         put "WARNING: Could not remove DS_VCHAR from &path";
         put 'See log for details';
         file log;
       run;
       %put WARNING: Could not assign libname to "&path";
       %put WARNING: Could not remove DS_VCHAR from "&path";
     %end;
   %else
     %do;
       %if %sysfunc(exist(reposmgr.DS_VCHAR)) %then
        %do;
          %put NOTE: Deleting unused DS_VCHAR from "&path";
          %let haveREPOSMGR=0; /* only do this once */
          proc datasets lib=reposmgr nodetails;
            delete DS_VCHAR;
          quit;
        %end;
       libname reposmgr clear;
     %end;
 %end;

/* If Reorg then:
   1) proc copy src library to dest library to Reorg (remove deleted records)
   2) switch srcDirPath and destDirPath so OS copy will replace main files with Reorged backup
*/

%put;
%put NOTE: Beginning &TYPE of repository files;
%put;

%let srcassgn=0;
%if &Reorg=Y %then  /* Note Reorg will be N if this is a Restore */
  %do;
    %put;
    %put NOTE: &TYPE Source Path is "&srcDirPath";
    %put NOTE: &TYPE Destination Path is "&destDirPath";
    %put;
    %let copyerr=1;
    libname src "%unquote(&srcDirPath)";
    %if &syslibrc^=0 %then
      %do;
       %put WARNING: Could not assign libname to "&srcDirPath";
       %goto errlib;
      %end;
    %let srcassgn=1;
    libname dest "%unquote(&destDirPath)";
    %if &syslibrc=0 %then
      %do;
        %if &createdDestDir=0 %then
         %do;
           %put;
           %put NOTE: First delete any existing files in the destination.;
           %put;
           proc datasets library=dest nodetails kill;
         %end;
        %put;
        %put NOTE: Copy repository files to destination.;
        %put;
        proc copy in=src out=dest; run;
        %if &syserr=0 or &syserr=4 %then
          %do;
            %let copyerr=0;
            %let temp=%superq(srcDirPath);
            %let srcDirPath=%superq(destDirPath);
            %let destDirPath=%superq(temp);
            /* Since server has lock on DS_DEFN in REPOSMGR - we update
               the destination copy of it. It will not get copied in Reorg
                on UNIX or Windows because of server lock */
            %if %superq(repositoryName)=REPOSMGR %then
              %do;
                 %*---------------------------------------------;
                 %* Get max ID count from DS_DEFN      ;
                 %*---------------------------------------------;
                 %let maxid=0;
                 proc sql NOPRINT;
                    select put(max(input(id,pib4.)),10.) into :maxid from dest.DS_DEFN;
                    quit;

                 %*---------------------------------------------------;
                 %* If DS_VCHAR exists then merge DS_DEFN and DS_VCHAR;
                 %* into the DS_DEFN dataset.  Only merge in the      ;
                 %* attributes: id, ds_name, var_name, var_chars and  ;
                 %* lots_of_dups from DS_VCHAR. Delete DS_VCHAR so it ;
                 %* does not get copied back to the real repository.  ;
                 %*---------------------------------------------------;
                 %if %sysfunc(exist(dest.DS_VCHAR))
                 %then %do;
                    proc sort data=dest.DS_DEFN; by ds_name var_name;run;
                    data dest.DS_DEFN;
                      update dest.DS_DEFN
                      dest.DS_VCHAR(keep=id ds_name var_name var_chars lots_of_dups);
                      by ds_name var_name;
                      drop mid;
                      if id = put(0,pib4.) then do;
                        mid = symget('maxid') + 1;
                        call symput('maxid',put(mid,10.));
                        id = put(mid,pib4.);
                      end;
                    run;
                    proc datasets lib=dest nodetails;
                      delete DS_VCHAR;
                    quit;
                 %end;
                 %*---------------------------------------------;
                 %* Update never_index field in DS_DEFN dataset ;
                 %*---------------------------------------------;
                 data dest.DS_DEFN; set dest.DS_DEFN;
                    if idx_used = 0 and idx_rjct > 10000 then never_index='Y';
                    if var_name='CLASNAME' then never_index='Y';
                    if var_name='LDESC' then never_index='Y';
                    if var_name='PREDROLE' then never_index='Y';
                    if var_name='SUCCROLE' then never_index='Y';
                    if var_name='ASCMGRID' then never_index='Y';
                 run;
              %end;
          %end;
        libname dest clear;
      %end;
    %else
     %do;
       %put WARNING: Could not assign libname to "&destDirPath";
     %end;
%errlib:
    %put ;
    %put ;
   %if &copyerr=1 %then
    %do;
      data _null_;
        file print;
        put "ERROR: Could not REORGANIZE repository &repositoryName";
        put 'See log for details';
        file log;
      run;
      %put WARNING: Prior error prevented REORGANIZE of repository.;
      %put ERROR: Could not REORGANIZE repository "&repositoryName";
    %end;
   %else
     %put NOTE: Reorganize complete. Preparing to copy Reorged files back to source repository;
   %if &srcassgn=0 %then
      libname src clear;
  %end; /* if &Reorg=Y */

/* ===  EDIT COPY COMMANDS BEGINNING HERE === */

%if &copyerr ne 1 %then
%do;
 /* Either do OS copy or proc copy.  OS copies will not work on the */
 /* workspace server in SP4 due to security issues associated with */
 /* submitting x commands.   Default for SP4 is proc copy. */
 %let USE_OS_COPIES=0; /* Set USE_OS_COPIES to determine how to copy */
                       /* 0 = Use proc copy                          */
                       /* 1 = Use OS commands via host x command. Not*/
                       /*     validate for use with SAS/MC.          */

 %put;
 %put NOTE: &TYPE Source Path is "&srcDirPath";
 %put NOTE: &TYPE Destination Path is "&destDirPath";
 %put;

 /* Don't copy REPOSMGR DS_DEFN back to repository location on */
 /* any platform due to lock contention. */
 %let isNOTREPOSMGR=0;
 %if ((&Restore=Y OR &Reorg=Y) AND %superq(repositoryName)=REPOSMGR) %then
   %do;
       %put NOTE: DS_DEFN not copied back to REPOSMGR directory.;
       %let isNOTREPOSMGR=1;
   %end;

 %if &USE_OS_COPIES eq 1 %then
  %do; /* Use OS commands */
   %if &sysscp=WIN OR &sysscp=DNTHOST %then
    %do;
       %if &sysscpl=WIN_NTSV %then
          %let cmd= xcopy  "&srcDirPath" "&destDirPath" /E /C /I /H /K;
       %else
          %let cmd= xcopy  "&srcDirPath" "&destDirPath" /E /C /I /H /K /O /X /Y;
       %put x &cmd;
       x &cmd;
       %let mysysrc=&sysrc;
       %if &sysrc = 0 %then
        %do;
          data _null_;
            file print;
            put "NOTE: &TYPE of &repositoryName Completed Successfully";
            file log;
          run;
          %put NOTE: Command completed successfully.;
          %put NOTE: &TYPE of "&repositoryName" Completed Successfully;
        %end;
    %end;
   %else %if &sysscp=OS %then
     %do;
       %let cmd="rm -fr '&destDirPath'";
       %put cmd=&cmd;
       %put filename cpbkup pipe &cmd;
       filename cpbkup pipe &cmd;
       data _NULL_;
         infile cpbkup;
         input;
         put _infile_;
       run;
       %let cmd="cp -pr '&srcDirPath' '&destDirPath' 2>&&1 | tee '&repositoryName._err'";
       %put cmd=&cmd;
       %put filename cpbkup pipe &cmd;
       filename cpbkup pipe &cmd;
       data _NULL_;
        infile cpbkup;
        input;
        put _infile_;
       run;
       %let mysysrc=&sysrc;
       %if &sysrc = 0 %then
        %do;
          %put NOTE: Command completed successfully.;
          %let cmd="rm -f '&repositoryName._err'";
          %put cmd=&cmd;
          %put filename cpbkup pipe &cmd;
          filename cpbkup pipe &cmd;
          data _NULL_;
            infile cpbkup;
            input;
            put _infile_;
          run;
        %end;
     %end;
   %else  /* assume UNIX or UNIX compatible */
     %do;
       %let cmd="rm -fr '&destDirPath'";
       %put x &cmd;
       x &cmd;
       %let cmd="cp -pr '&srcDirPath' '&destDirPath' 2>&&1 | tee '&repositoryName._err'";
       x &cmd;
       %put ;
       %let mysysrc=&sysrc;
       %if &sysrc = 0 %then
        %do;
          data _null_;
            file print;
            put "NOTE: &TYPE of &repositoryName Completed Successfully";
            file log;
          run;
          %put NOTE: Command completed successfully.;
          %put NOTE: &TYPE of "&repositoryName" Completed Successfully;
          %let cmd="rm -f '&repositoryName._err'";
          x &cmd;
        %end;
     %end;
   %put ;
   %put ;
   /* Report copy errors on any of the platforms */
   %if &mysysrc ne 0 %then
     %do;
       %if &Reorg=Y and &copyerr=0 %then
        %do;
          data _null_;
            file print;
            put "ERROR: Could not &TYPE repository &repositoryName";
            put "ERROR: Cannot RESUME repository &repositoryName";
            put 'See log for details';
            file log;
          run;
          %put WARNING: Prior error prevented REORGANIZE of repository.;
          %put ERROR: Could not REORGANIZE repository "&repositoryName";
          %put ERROR: Cannot RESUME repository "&repositoryName";
          %let reorgerr=1;
        %end;
       %else
        %do;
          data _null_;
            file print;
            put "ERROR: Could not &TYPE repository &repositoryName";
            put 'See log for details';
            file log;
          run;
          %put ERROR: Command returned error code &sysrc, see "&repositoryName._err";
          %let cmd="cat &repositoryName._err";
          x &cmd;
          %put ;
          %put WARNING: Prior error prevented &TYPE of repository.;
          %put ERROR: Could not &TYPE repository "&repositoryName";
        %end;
     %end;
  %end;
 %else
  %do; /* Use proc copy */
   %let copyerr=1;
   libname src "%unquote(&srcDirPath)";
   %if &syslibrc=0 %then
     %do;
       libname dest "%unquote(&destDirPath)";
       %if &syslibrc=0 %then
         %do;
           %if &createdDestDir=0 %then
            %do;
              %if &isNOTREPOSMGR ne 1 %then
                %do;
                 %put;
                 %put NOTE: First delete any existing files in the destination.;
                 %put;
                 proc datasets library=dest nodetails kill;
                %end;
            %end;
           %put;
           %put NOTE: Copy repository files to destination.;
           %put;
           %if &isNOTREPOSMGR ne 1 %then
             %do;
                proc copy in=src out=dest; run;
             %end;
           %else
             %do;
                proc copy in=src out=dest; exclude DS_DEFN; run;
             %end;
           %if &syserr=0 or &syserr=4 %then
             %do;
               %let copyerr=0;
             %end;
           libname dest clear;
         %end;
       %else
          %put WARNING: Could not assign libname to "&destDirPath";
       libname src clear;
     %end;
   %else
      %put WARNING: Could not assign libname to "&srcDirPath";
  %put;
  %put;
  %if &copyerr=1 %then
   %do;
     %if &Reorg=Y %then
      %do;
        data _null_;
          file print;
          put "ERROR: Could not &TYPE repository &repositoryName";
          put "ERROR: Cannot RESUME repository &repositoryName";
          put 'See log for details';
          file log;
        run;
        %put WARNING: Prior error prevented REORGANIZE of repository.;
        %put ERROR: Could not REORGANIZE repository "&repositoryName";
        %put ERROR: Cannot RESUME repository "&repositoryName";
        %let reorgerr=1;
      %end;
     %else
      %do;
        data _null_;
          file print;
          put "ERROR: Could not &TYPE repository &repositoryName";
          put 'See log for details';
          file log;
        run;
        %put WARNING: Prior error prevented &TYPE of repository.;
        %put ERROR: Could not &TYPE repository "&repositoryName";
      %end;
   %end;
 %else
   %do;
      data _null_;
        file print;
        put "NOTE: &TYPE of &repositoryName Completed Successfully";
        file log;
      run;
      %put NOTE: Command completed successfully.;
      %put NOTE: &TYPE of "&repositoryName" Completed Successfully;
   %end;
 %end;  /* if USE_OS_COPIES */
%end;  /* if &copyerr ne 1 */
%put;

%nocopy:
%mend;

/* === EDIT DEFAULT OMABAKUP PATHNAMES BEGINNING HERE === */
/* === CHANGE VALUE OF &PLAN MACRO VARIABLE TO THE PLAN NAME   === */

%macro omabudef();
%global PLAN;
%global defaultDestinationPath;
%global defaultServerStartPath;
%global defaultRposmgrPath;

%let PLAN=9.1; /* ==== CHANGE TO PLAN NAME ==== */

%if &sysscp=WIN OR &sysscp=DNTHOST %then
 %do;
  %let defaultDestinationPath=C:\SAS\&PLAN\Lev1\SASBackup;
  %let defaultServerStartPath=C:\SAS\&PLAN\Lev1\SASMain;
  %let defaultRposmgrPath=MetadataServer\rposmgr;
 %end;
%else
 %do;
   %if &sysscp=OS %then
    %do;
      %let defaultDestinationPath=/usr/lpp/SAS/&PLAN/Lev1/SASBackup;
      %let defaultServerStartPath=/usr/lpp/SAS/&PLAN/Lev1/SASMain;
      %let defaultRposmgrPath=MetadataServer/rposmgr;
    %end;
   %else
    %do;
      %let HOME=%sysget(HOME);
      %let defaultDestinationPath=&HOME/SAS/&PLAN/Lev1/SASBackup;
      %let defaultServerStartPath=&HOME/SAS/&PLAN/Lev1/SASMain;
      %let defaultRposmgrPath=MetadataServer/rposmgr;
    %end;
 %end;
%mend;
%omabudef(); /* Include these defaults */

/* === NO USER CHANGES SHOULD OCCUR BELOW THIS POINT  === */

%macro OMABUINI();
/* Setup some XML Maps that we need for parsing metadata XML */
filename OMABUF1 catalog 'backdest.OMA_Backup.getRepositories.source';
filename OMABUF1m catalog 'backdest.OMA_Backup.getRepositoriesMap.source';
%put "DEBUG: 913 XML ENCODING";
data _null_;
   file OMABUF1m;
   length enc $40;
   enc = getoption("ENCODING");
put '<?xml version="1.0" encoding="' enc +(-1) '" ?>';
put '<SXLEMAP version="1.2"> <TABLE name="REPOSITORIES"> <TABLE-PATH syntax="xpath"> /GetRepositories/Repositories/Repository </TABLE-PATH>';
put '<COLUMN name="id"> <PATH> /GetRepositories/Repositories/Repository/@Id </PATH> <TYPE>character</TYPE> <DATATYPE>STRING</DATATYPE> <LENGTH>17</LENGTH> </COLUMN>'; 
put '<COLUMN name="name"> <PATH> /GetRepositories/Repositories/Repository/@Name </PATH> <TYPE>character</TYPE> <DATATYPE>STRING</DATATYPE> <LENGTH>60</LENGTH> </COLUMN>';
put '<COLUMN name="desc"> <PATH> /GetRepositories/Repositories/Repository/@Desc </PATH> <TYPE>character</TYPE> <DATATYPE>STRING</DATATYPE> <LENGTH>80</LENGTH> </COLUMN>';
put '<COLUMN name="defaultns"> <PATH> /GetRepositories/Repositories/Repository/@DefaultNS </PATH> <TYPE>character</TYPE> <DATATYPE>STRING</DATATYPE> <LENGTH>8</LENGTH> </COLUMN>';
put '<COLUMN name="access"> <PATH> /GetRepositories/Repositories/Repository/@Access </PATH> <TYPE>character</TYPE> <DATATYPE>STRING</DATATYPE> <LENGTH>20</LENGTH> </COLUMN>';
put '<COLUMN name="pauseState"> <PATH> /GetRepositories/Repositories/Repository/@PauseState </PATH> <TYPE>character</TYPE> <DATATYPE>STRING</DATATYPE> <LENGTH>8</LENGTH> </COLUMN>';
put '</TABLE> </SXLEMAP>';
run;

filename OMABUF2  catalog 'work.OMA_Backup.getRepository.source';
filename OMABUF2m catalog 'work.OMA_Backup.getRepositoryMap.source';
data _null_;
   file OMABUF2m;
   length enc $40;
   enc = getoption("ENCODING");
put '<?xml version="1.0" encoding="' enc +(-1) '" ?>';
put '<SXLEMAP version="1.2"> <TABLE name="REPOSITORY"> <TABLE-PATH syntax="xpath"> /GetMetadata/Metadata/RepositoryBase </TABLE-PATH>';
put '<COLUMN name="id"> <PATH> /GetMetadata/Metadata/RepositoryBase/@Id </PATH> <TYPE>character</TYPE> <DATATYPE>STRING</DATATYPE> <LENGTH>17</LENGTH> </COLUMN>';
put '<COLUMN name="name"> <PATH> /GetMetadata/Metadata/RepositoryBase/@Name </PATH> <TYPE>character</TYPE> <DATATYPE>STRING</DATATYPE> <LENGTH>60</LENGTH> </COLUMN>';
put '<COLUMN name="desc"> <PATH> /GetMetadata/Metadata/RepositoryBase/@Desc </PATH> <TYPE>character</TYPE> <DATATYPE>STRING</DATATYPE> <LENGTH>80</LENGTH> </COLUMN>';
put '<COLUMN name="path"> <PATH> /GetMetadata/Metadata/RepositoryBase/@Path </PATH> <TYPE>character</TYPE> <DATATYPE>STRING</DATATYPE> <LENGTH>200</LENGTH> </COLUMN>';
put '</TABLE> </SXLEMAP>';
run;
%mend;

%macro OMABUGRS();
/* get list of all repositories and their current state from server */
PROC METADATA IN=' <GetRepositories> <Repositories/> <Flags>1</Flags>  <!-- OMI_ALL  --> <Options/> </GetRepositories> ' out=OMABUF1;
RUN;

libname OMABUF1 xml xmlmap=OMABUF1m;
proc print data=OMABUF1.repositories;
   var name id access pauseState;
run;

%mend;

/* SAS Metadata Server Backup Repository */
%macro OMABUREP(name,id,DestinationPath,ServerStartPath,RposmgrPath,
                                Reorg=Y,Restore=N);


%let name=%qcmpres(&name);
%let DestinationPath=%qcmpres(&DestinationPath);
%let ServerStartPath=%qcmpres(&ServerStartPath);
%let RposmgrPath=%qcmpres(&RposmgrPath);

/*
%put DEBUG: OMABUREP: Name=&name;
%put DEBUG: OMABUREP: Id=&id;
%put DEBUG: OMABUREP: DestinationPath=&DestinationPath;
%put DEBUG: OMABUREP: SourcePath=&ServerStartPath;
%put DEBUG: OMABUREP: RposmgrPath=&RposmgrPath;
%put DEBUG: OMABUREP: Reorg=&Reorg;
%put DEBUG: OMABUREP: Restore=&Restore;
*/

%let assignOMABUF2=0;
/* Repository manager repository is a special case. */
%if %superq(name)=REPOSMGR %then
 %do;
   %put;
   %put **********************************************************************;
   %put                     BEGIN &TITLETYPE OF "&name";
   %put **********************************************************************;
   %put;
   %let isREPOSMGR=1;
   /* Set default REPOSMGR */
   %if %superq(RposmgrPath)=%str() %then
   %do;
      %if &sysscp=WIN OR &sysscp=DNTHOST %then
        %let RposmgrPath=MetadataServer\rposmgr;
      %else
        %let RposmgrPath=MetadataServer/rposmgr;
   %end;
   %let reposName=&name;
   %let repositoryId=&id;
   %let reposPath=%superq(RposmgrPath);
   %put Preparing &TYPE of Name=&reposName  Id=&repositoryId  Path=&reposPath;
   %put;
   data OMABUF2t;
     id=symget('id');
     name=symget('name');
     desc='The Repository Manager';
     path=symget('reposPath');
   run;
   proc print data=OMABUF2t;
   run;
 %end;
%else
 %do;
   %let isREPOSMGR=0;
   /* For all other repositories, we need to query the repository */
   /* manager to discern the path. */
   PROC METADATA
   IN=" <GetMetadata> <Metadata> <RepositoryBase Name='' Id='&id' Path=''/>
   </Metadata> <Flags>1</Flags>  <!-- OMI_ALL  --> <Ns>REPOS</Ns>
   <Options/> </GetMetadata> " out=OMABUF2;
   RUN;
   %let assignOMABUF2=1;
   libname OMABUF2 xml xmlmap=OMABUF2m;
   proc print data=OMABUF2.repository;
   run;

   /* Translate to underscores, any special characters contained within  */
   /* the registered repository name, that cannot be used to create the  */
   /* backup destination directory name.  This processing is divided     */
   /* into WIN, and all the other platforms.  You cannot create a folder */
   /* on WIN that contains '\/:"*?<>|'. UNIX boxes will not let you      */
   /* create a directory that contains '/'. Additionally, the data step  */
   /* function dcreate, won't create a directory if the pathname contains*/
   /* a '*' and the libname statement won't assign a libref to a pathname*/
   /* that contains a '\'; Also, macro doesn't process exising pathnames */
   /* very well that contain double and/or single quotes. */

   %put;
   %put **********************************************************************;
   %put                     BEGIN &TITLETYPE OF "&name";
   %put **********************************************************************;
   %put;
   %if &sysscp=WIN OR &sysscp=DNTHOST %then
    %do;
      data _null_;
       set OMABUF2.repository;
       call symputx("reposName",translate(name,'____________',"%str(/:%'%"[]*?<>|\)"));
       call symputx("repositoryId",id);
       call symputx("reposPath",path);
       backuptype=symget('TYPE');
       put 'Preparing ' backuptype 'of Name=' name '  Id=' id ' Path=' path;
       put '.';
      run;
    %end;
   %else /* UNIX */
    %do;
     data _null_;
      set OMABUF2.repository;
      call symputx("reposName",translate(name,'________',"%str(/%'%"[]*?\)"));
      call symputx("repositoryId",id);
      call symputx("reposPath",path);
      backuptype=symget('TYPE');
       put 'Preparing ' backuptype 'of Name=' name '  Id=' id ' Path=' path;
      put '';
      run;
     %end;
 %end;

%let reposPath=%qcmpres(&reposPath);
%let reposName=%qcmpres(&reposName);


%put;
%put NOTE: Begin validation of source and destination pathnames;
%put;

%if &sysscp=WIN OR &sysscp=DNTHOST %then
 %do;
   /* Clean-up windows slashes */
   %let temp=%sysfunc(translate(&reposPath,'\','/'));
   %let srcReposPath=%superq(temp);
   %let temp=%sysfunc(translate(&DestinationPath,'\','/'));
   %let DestinationPath=%superq(temp);
   %let temp=%sysfunc(translate(&ServerStartPath,'\','/'));
   %let ServerStartPath=%superq(temp);
   %let temp=%sysfunc(translate(&reposName,'\','/'));
   %let repositoryName=%superq(temp);
   %let destReposPath=&DestinationPath\&repositoryName;
 %end;
%else
 %do;
   %let temp=%sysfunc(translate(&reposPath,'/','\'));
   %let srcReposPath=%superq(temp);
   %let temp=%sysfunc(translate(&DestinationPath,'/','\'));
   %let DestinationPath=%superq(temp);
   %let temp=%sysfunc(translate(&ServerStartPath,'/','\'));
   %let ServerStartPath=%superq(temp);
   %let temp=%sysfunc(translate(&reposName,'/','\'));
   %let repositoryName=%superq(temp);
   %let destReposPath=&DestinationPath/&repositoryName;
 %end;


/*
%put;
%put DEBUG: OMABUREP Paths and names after translation;
%put;
%put DEBUG: OMABUREP: srcReposPath=&srcReposPath;
%put DEBUG: OMABUREP: DestinationPath=&DestinationPath;
%put DEBUG: OMABUREP: repositoryName=&repositoryName;
%put DEBUG: OMABUREP: destReposPath=&destReposPath;
%put;
*/

/* Capture the dest path before appending respository dest name */
%let destPath=&DestinationPath;

/* If Restore, swap the destination with the source */
/* Munge the paths around in preparation for validation */
%let addRepositoryName=%str();
/* Save registered name */
%let registeredName=&repositoryName;
%if &Restore=Y %then
  %do;
     %let tempDest=&srcReposPath;
     %let tempReposName=&repositoryName;
     /* Is this a simple dir name?  Look for a path delimiter */
     %let i=0;
     %let addRepositoryName=%str();
     %if &sysscp=WIN OR &sysscp=DNTHOST %then
        %let i=%index(&srcReposPath,%str(\));
     %else
        %let i=%index(&srcReposPath,%str(\));
     %if &i gt 0 %then
       %do; /* Paren, remove the repository name from the path */
         %if &sysscp=WIN OR &sysscp=DNTHOST %then
           %do;
              data _null_;
               p=symget('srcReposPath');
               pr=reverse(p);
               x=index(pr,'\');
               /* Get the repos path w/o the repos name */
               rpr=substr(pr,x+1,length(pr)-x);
               rp=reverse(rpr);
               call symputx("tempDest",trim(rp));
               /* Get the repos name w/o the repos path */
               rnr=substr(pr,1,x-1);
               rn=reverse(rnr);
               call symputx("tempReposName",trim(rn));
             run;
           %end;
         %else
           %do; /* Unix plus */
              data _null_;
               p=symget('srcReposPath');
               pr=reverse(p);
               x=index(pr,'/');
               /* Get the repos path w/o the repos name */
               rpr=substr(pr,x+1,length(pr)-x);
               rp=reverse(rpr);
               call symputx("tempDest",trim(rp));
               /* Get the repos name w/o the repos path */
               rnr=substr(pr,1,x-1);
               rn=reverse(rnr);
               call symputx("tempReposName",trim(rn));
             run;
           %end;
       %end;
     %else
       %do; /* No path delimiter, we (likely) have a simple repository name */
            /* so don't add it again. */
         %let addRepositoryName=N;
       %end;

    /* Restore is a copy back to the registered repository location, so */
    /* always use the registered repository name in the destination. */
    %let registeredName=%superq(repositoryName);
    %let repositoryName=%superq(tempReposName);
    %let srcReposPath=%superq(destReposPath);
    %let DestinationPath=%superq(tempDest);

/*
%put;
%put DEBUG: OMABUREP: Paths and names prepared for RESTORE;
%put;
%put DEBUG: OMABUREP: srcReposPath=&srcReposPath;
%put DEBUG: OMABUREP: DestinationPath=&DestinationPath;
%put DEBUG: OMABUREP: repositoryName=&repositoryName;
%put DEBUG: OMABUREP: registeredName=&registeredName;
%put DEBUG: OMABUREP: addRepositoryName=&addRepositoryName;
%put;
*/

  %end;

/* Validate repository and destination paths */
/* First check for blank repository -- actually should never happen */
%if &srcReposPath = %str() %then
 %do;
   data _null_;
     file print;
     put "ERROR: Could not &TYPE repository &repositoryName";
     put 'See log for details';
     file log;
   run;
   %put WARNING: Found blank repository path "&srcReposPath";
   %put "ERROR: Could not &TYPE REPOSMGR &repositoryName";
   %goto abortout;
 %end;
%else
 %do;
   /* Validate DestinationPath is an existing directory */
   %put NOTE: Testing destination "&DestinationPath" for relative path;
   %let testDestPath=%superq(DestinationPath);
   %if &sysscp=WIN OR &sysscp=DNTHOST %then
      %let testRelativePath=%nrbquote(&ServerStartPath\&testDestPath);
   %else
      %let testRelativePath=%nrbquote(&ServerStartPath/&testDestPath);
   %if %sysfunc(fileexist("&testRelativePath")) %then
    %do;
      %put NOTE: Replacing relative repository path with absolute path using server start path.;
      %let testDestPath=%superq(testRelativePath);
      %put NOTE: Relative repository path resolved to absolute path &testDestPath.;
    %end;
   %else
    %do;
      %put NOTE: Assume registered path is already absolute.;
    %end;
   %put;
   /* Test dir exist with libname b/c fileexist finds files too */
   %put NOTE: Testing that destination "&DestinationPath" exists;
   libname destpath "%unquote(&testDestPath)";
   %if &syslibrc ne 0 %then
    %do;
      %put WARNING: Could not assign libname to "&testDestPath";
      %put WARNING: Could not locate Destination path "&testDestPath";
      %if &isREPOSMGR eq 1 %then
       %do;
          data _null_;
            file print;
            put "ERROR: Could not &TYPE REPOSMGR &repositoryName";
            put 'See log for details';
            file log;
          run;
          %put WARNING: Prior error prevented &TYPE of repository.;
          %put ERROR: Could not &TYPE REPOSMGR "&repositoryName";
       %end;
      %else
       %do;
          data _null_;
            file print;
            put "ERROR: Could not &TYPE repository &repositoryName";
            put 'See log for details';
            file log;
          run;
          %put WARNING: Prior error prevented &TYPE of repository.;
          %put ERROR: Could not &TYPE repository "&repositoryName";
       %end;
      libname destpath clear;
    %end;
   %else
    %do;
     libname destpath clear;
     %let DestinationPath=%superq(testDestPath);
     %put;
     /* Some repository paths are relative to the server startup           */
     /* directory. Try that first, if it works as a SAS library, use that. */
     %put NOTE: Testing source "&srcReposPath" for relative path;
     %if &addRepositoryName^=N %then
        %let addRepositoryName=Y;
     %if &sysscp=WIN OR &sysscp=DNTHOST %then
        %let testRelativePath=%nrbquote(&ServerStartPath\&srcReposPath);
     %else
        %let testRelativePath=%nrbquote(&ServerStartPath/&srcReposPath);
     %if %sysfunc(fileexist("&testRelativePath")) %then
      %do;
        %put NOTE: Replacing relative repository path with absolute path using server start path.;
        %let srcReposPath=%superq(testRelativePath);
        %put NOTE: Relative repository path resolved to absolute path &srcReposPath.;
        %put;
        %let testRelativeDestPath=%superq(DestinationPath);
/*
        %put NOTE: Testing destination &testRelativeDestPath for relative path;
*/
        %if &sysscp=WIN OR &sysscp=DNTHOST %then
           %let testRelativeDestPath=%nrbquote(&DestinationPath\&srcReposPath);
        %else
           %let testRelativeDestPath=%nrbquote(&DestinationPath/&srcReposPath);
        %if %sysfunc(fileexist("&testRelativeDestPath")) %then
         %do;
           %put NOTE: Relative path exists in destination directory as well.;
           %let DestinationPath=%superq(testRelativeDestPath);
           %put NOTE: Relative destination path resolved to absolute path &DestinationPath.;
           %let addRepositoryName=N;
         %end;
      %end;
     %else
      %do;
        %put NOTE: Assume registered path is already absolute.;
      %end;
     %put;
     /* Validate this path really points to a valid repository by checking  */
     /* for key datasets mrrgstry and cntainer.                             */
     %put NOTE: Testing that source "&srcReposPath" exists;
     libname srcpath "%unquote(&srcReposPath)";
     %if &syslibrc ne 0 %then
      %do;
        %put WARNING: Could not assign libname to "&srcReposPath";
        %if &Restore=N %then
          %do;
           %put WARNING: Could not locate source path "&srcReposPath";
           %if &isREPOSMGR eq 1 %then
            %do;
               data _null_;
                 file print;
                 put "ERROR: Could not &TYPE REPOSMGR &repositoryName";
                 put 'See log for details';
                 file log;
               run;
               %put WARNING: Prior error prevented &TYPE of repository.;
               %put ERROR: Could not &TYPE REPOSMGR "&repositoryName";
            %end;
           %else
            %do;
               data _null_;
                 file print;
                 put "ERROR: Could not &TYPE repository &repositoryName";
                 put 'See log for details';
                 file log;
               run;
               %put WARNING: Prior error prevented &TYPE of repository.;
               %put ERROR: Could not &TYPE repository "&repositoryName";
            %end;
          %end;
        %else /* Restore */
          %do;
           %put WARNING: Repository "&repositoryName" does not exist in backup location "&srcReposPath";
           data _null_;
              file print;
              put "WARNING: Repository &repositoryName does not exist in backup location &srcReposPath";
              put "WARNING: Could not &TYPE repository &repositoryName";
              put 'See log for details';
              file log;
           run;
           %put WARNING: Prior error prevented &TYPE of repository.;
           %put WARNING: Could not &TYPE repository "&repositoryName";
          %end;
       libname srcpath clear;
      %end;
     %else
      %do;
       %put;
       %put NOTE: Validate "&srcReposPath" is a repository;
       %let dsname=srcpath.mrrgstry;
       %if %sysfunc(exist(&dsname)) %then
        %do;
          %put NOTE: Located MRRGSTRY container.;
          %let dsname=srcpath.cntainer;
          %if %sysfunc(exist(&dsname)) %then
           %do;
             %put NOTE: Located CNTAINER container.;
             libname srcpath clear;
             %put;
             %put NOTE: End validation of source and destination pathnames;
             %put;
             %OMABUCPY(%superq(repositoryName),
                     &registeredName,
                     &repositoryId,
                     %superq(srcReposPath),
                     %superq(DestinationPath),
                     %superq(destPath),
                     addRepositoryName=&addRepositoryName,
                     Reorg=&Reorg,
                     Restore=&Restore)
           %end;
          %else
           %do;
             %put WARNING: Could not locate CNTAINER container.;
             %if &isREPOSMGR eq 1 %then
              %do;
                 data _null_;
                   file print;
                   put "ERROR: Could not &TYPE REPOSMGR &repositoryName";
                   put 'See log for details';
                   file log;
                 run;
                 %put WARNING: Prior error prevented &TYPE of repository.;
                 %put ERROR: Could not &TYPE REPOSMGR "&repositoryName";
              %end;
             %else
              %do;
               data _null_;
                 file print;
                 put "ERROR: Could not &TYPE repository &repositoryName";
                 put 'See log for details';
                 file log;
               run;
               %put WARNING: Prior error prevented &TYPE of repository.;
               %put ERROR: Could not &TYPE repository "&repositoryName";
              %end;
             libname srcpath clear;
             %put;
             %put NOTE: End validation of source and destination pathnames;
             %put;
           %end;
        %end;
       %else
        %do;
          %put WARNING: Could not locate MRRGSTRY container.;
          %if &isREPOSMGR eq 1 %then
           %do;
             data _null_;
               file print;
               put "ERROR: Could not &TYPE REPOSMGR &repositoryName";
               put 'See log for details';
               file log;
             run;
             %put WARNING: Prior error prevented &TYPE of repository.;
             %put ERROR: Could not &TYPE REPOSMGR "&repositoryName";
           %end;
          %else
           %do;
             data _null_;
               file print;
               put "ERROR: Could not &TYPE repository &repositoryName";
               put 'See log for details';
               file log;
             run;
             %put WARNING: Prior error prevented &TYPE of repository.;
             %put ERROR: Could not &TYPE repository "&repositoryName";
           %end;
          libname srcpath clear;
          %put;
          %put NOTE: End validation of source and destination pathnames;
          %put;
        %end;
      %end;
   %end; /* if &syslibrc ne 0 */
 %end; /*  if %superq(srcReposPath) = %str()  */
%abortout:
%put;
%put **********************************************************************;
%put                     END &TITLETYPE OF "&name";
%put **********************************************************************;
%put;
%mend;

%macro BACKLIB(reposname);
%if &sysscp=WIN OR &sysscp=DNTHOST %then
  libname BACKLIB "%unquote(&destinationPath\&reposname)";
%else
  libname BACKLIB "%unquote(&destinationPath/&reposname)";
%mend;

/* Setup complete.  Now start the real work. */
%macro OMABAKUP(
          DestinationPath=&defaultDestinationPath,
          ServerStartPath=&defaultServerStartPath,
          RposmgrPath=&defaultRposmgrPath,
          Reorg=N,Restore=N,RunAnalysis=N,
          RepositoryList=ALL
          );

%global TYPE;
%global reorgerr;
%let reorgerr=0;
%global assignOMABUF2;
%global haveREPOSMGR;
%global haveAllRepositories;
%let assignOMABUF2=0;
%let haveREPOSMGR=0;
%let haveAllRepositories=0;

/* Clean-up inputs, assign defaults */
/* DestinationPath */
%let DestinationPath=%superq(DestinationPath);
%if %length(&DestinationPath)>0 %then
 %do;
  %let temp=%sysfunc(translate(&DestinationPath, %str(  ),%str(%'%")));
  %let DestinationPath=%qcmpres(&temp);
 %end;
%if DestinationPath=%str() %then
  %let DestinationPath=&defaultDestinationPath;

/* ServerStartPath */
%let ServerStartPath=%superq(ServerStartPath);
%if %length(&ServerStartPath)>0 %then
 %do;
  %let temp=%sysfunc(translate(&ServerStartPath, %str(  ),%str(%'%")));
  %let ServerStartPath=%qcmpres(&temp);
 %end;
%if &ServerStartPath=%str() %then
  %let ServerStartPath=&defaultServerStartPath;

/* RposmgrPath */
%let RposmgrPath=%superq(RposmgrPath);
%if %length(&RposmgrPath)>0 %then
 %do;
  %let temp=%sysfunc(translate(&RposmgrPath, %str(  ),%str(%'%")));
  %let RposmgrPath=%qcmpres(&temp);
 %end;
%if &RposmgrPath=%str() %then
  %let RposmgrPath=&defaultRposmgrPath;

/* Reorg */
%let Reorg=%sysfunc(translate(&Reorg,"YESNO","yesno"));
%if %length(&Reorg)>0 %then
 %do;
  %let temp=%sysfunc(translate(&Reorg, %str(  ),%str(%'%")));
  %let Reorg=%qcmpres(&temp);
 %end;
%if &Reorg=Y OR &Reorg=YES %then
   %let Reorg=Y;
%else
   %let Reorg=N;

/* Restore */
%let Restore=%sysfunc(translate(&Restore,"YESNO","yesno"));
%if %length(&Restore)>0 %then
 %do;
  %let temp=%sysfunc(translate(&Restore, %str(  ),%str(%'%")));
  %let Restore=%qcmpres(&temp);
 %end;
%if &Restore=Y OR &Restore=YES %then
   %let Restore=Y;
%else
   %let Restore=N;

/* RunAnalysis */
%let RunAnalysis=%sysfunc(translate(&RunAnalysis,"YESNO","yesno"));
%if %length(&RunAnalysis)>0 %then
 %do;
  %let temp=%sysfunc(translate(&RunAnalysis, %str(  ),%str(%'%")));
  %let RunAnalysis=%qcmpres(&temp);
 %end;
%if &RunAnalysis=Y OR &RunAnalysis=YES %then
   %let RunAnalysis=Y;
%else
   %let RunAnalysis=N;

/* RepositoryList */
%let RepositoryList=%superq(RepositoryList);
%let saveList=&RepositoryList;
%if %length(&RepositoryList)>0 %then
 %do;
   %let temp=%sysfunc(translate(&RepositoryList, %str(  ),%str(%'%")));
   %let RepositoryList=%superq(temp);
 %end;
%if &RepositoryList=%str() OR &RepositoryList=%str(all) OR
    &RepositoryList=%str('ALL') OR &RepositoryList=%str('all') OR
    &RepositoryList=%str("ALL") OR  &RepositoryList=%str(ALL) OR
    &RepositoryList=%str("all") %then
   %let RepositoryList=ALL;
%else
   %let RepositoryList=&saveList;

/* Get the pathnames cleaned up. */
%if &sysscp=WIN OR &sysscp=DNTHOST %then
 %do;
   /* Clean-up Windows slashes */
   %let temp=%sysfunc(translate(&DestinationPath,'\','/'));
   %let DestinationPath=%superq(temp);
   %let temp=%sysfunc(translate(&ServerStartPath,'\','/'));
   %let ServerStartPath=%superq(temp);
   %let temp=%sysfunc(translate(&RposmgrPath,'\','/'));
   %let RposmgrPath=%superq(temp);
 %end;
%else
 %do;
   /* Clean-up Unix slashes */
   %let temp=%sysfunc(translate(&DestinationPath,'/','\'));
   %let DestinationPath=%superq(temp);
   %let temp=%sysfunc(translate(&ServerStartPath,'/','\'));
   %let ServerStartPath=%superq(temp);
   %let temp=%sysfunc(translate(&RposmgrPath,'/','\'));
   %let RposmgrPath=%superq(temp);
 %end;

/* Turn off Reorg and runanalysis if we are doing a Restore */
%if &Restore=Y AND &Reorg=Y %then
  %do;
     %put NOTE: Restore setting Reorg to "N";
     %let Reorg=N;
  %end;
%if &Restore=Y AND &RunAnalysis=Y %then
  %do;
     %put NOTE: Restore setting RunAnalysis to "N";
     %let RunAnalysis=N;
  %end;

/* Establish what type of backup this is */
%if &Restore=N %then
  %do;
   %let TYPE=BACKUP;
   %if &Reorg=Y and &RunAnalysis=Y %then
     %let TITLETYPE=REORGANIZE with ANALYSIS BACKUP;
   %else %if &Reorg=Y %then
     %let TITLETYPE=REORGANIZE BACKUP;
   %else %if &RunAnalysis=Y %then
     %let TITLETYPE=ANALYSIS BACKUP;
   %else
     %let TITLETYPE=BACKUP;
  %end;
%else
  %do;
   %let TYPE=RESTORE;
   %let TITLETYPE=RESTORE;
  %end;

%put ;
%put ;
%put NOTE: OMABAKUP - &TYPE SAS Metadata Server;
%put NOTE: DestinationPath=&DestinationPath;
%put NOTE: ServerStartPath=&ServerStartPath;
%put NOTE: RposmgrPath=&RposmgrPath;
%put NOTE: Restore=&Restore;
%put NOTE: Reorg=&Reorg;
%put NOTE: RunAnalysis=&RunAnalysis;
%put NOTE: RepositoryList=&RepositoryList;
%put NOTE: sysscp=&sysscp;
%put NOTE: sysscpl=&sysscpl;
%put;


/* First, setup some macros and needed resources. */
title Prepare all repositories for &TITLETYPE operations;
footnote SAS Metadata Server Backup Utility;
options nocenter linesize=MAX;

%let testDestPath=%superq(DestinationPath);
/* Keep backdest libname for catalog source */
libname backdest "%unquote(&testDestPath)";
%if &syslibrc ne 0 %then
%do;
   data _null_;
     file print;
     put "ERROR: Could not locate destination &testDestPath. Repositories NOT processed";
     put 'See log for details';
     file log;
   run;
   %put WARNING: Could not assign libname to "&testDestPath";
   %put ERROR: Could not locate destination "&testDestPath". Repositories NOT processed;
   %goto getout;
%end;

%if &Restore=N %then
%do;
   /* Initialize null variable analysis job in OMAVARAN file and then
    * set OMAVARAN fileref to append mode. omabucpy will add statements to
    * this job for each repository it copies
    */
   filename OMAVARAN  catalog 'backdest.OMA_Varan.VarAnalyzeAll.source';
   data _null_;
      file OMAVARAN;
      put '/* This job will do a variable analyze on all the repositories backed up under it */';
      stop;
   run;
   filename OMAVARAN  catalog 'backdest.OMA_Varan.VarAnalyzeAll.source' mod;
%end;

%OMABUINI();

/* list server status - just because */
proc metaoperate action=status; run;
title Initial Repository States For &TITLETYPE;
%OMABUGRS();

/* Pause the REPOSMGR repository for READONLY in the server */
/* Nobody can add or delete repositories until we are done. */
proc metaoperate
     action=pause
     options='<REPOSITORY ID="REPOSMGR" STATE="READONLY"/>';
run;

/*
%put;
%put DEBUG: RepositoryList=&RepositoryList;
%put;
*/

/* Process repositories by name from the RepositoryList */
/* Find the repository names delimited by commas */
%let subsetIf=%str();
%let options=%str();
%let shortIdList=%str();
%let pauseSyntaxList=%str();
%let addREPOSMGR=1;
%let nRepositoryList=0;
%if %superq(RepositoryList)^=ALL %then
 %do;
   %let i=1;
   %let nameList=%qsubstr(&RepositoryList,2,%length(&RepositoryList)-2);
   %do %until (%superq(name)=%str());
      %let name=%qscan(&nameList,&i,%str(,));
      %if &name^=%str() %then
       %do;
         %let name=%qcmpres(&name);
         %if %superq(name) = REPOSMGR %then
          %do;
           %let addREPOSMGR=0;
           %let haveREPOSMGR=1;
          %end;
         %if &i eq 1 %then
          %do;
            %let subsetIf=name="&name";
          %end;
         %else
          %do;
            %let subsetIf=&subsetIf OR name="&name";
          %end;
         %let i=%eval(&i+1);
       %end;
   %end;
   %let nRepositoryList = %eval(&i-1);
   /* Add the REPOSMGR if this is type BACKUP */
   %if &Restore=N AND &addREPOSMGR eq 1 %then
    %do;
       %let subsetIf=&subsetIf OR name="REPOSMGR";
    %end;
 %end;
%else
 %do;
   %let haveREPOSMGR=1;
 %end;
%let subsetIf=%superq(subsetIf);
%if &subsetIf^=%str() %then
 %do;
   %let subsetIf=if &subsetIf;
  %end;
%else %if &Restore=Y %then
%do;
   %let subsetIf=if name="REPOSMGR" then DELETE;
%end;

/*
%put;
%put DEBUG: subsetIf=&subsetIf;
%put;
*/

/* In this datastep we will process the repositories on this server,
 * subsetting by the contents of the RepositoryList parameter by repository
 * name.  The pauseIdList is built as we go to be used by metaoperate
 * to pause and resume qualified repositories by ID.
 */
%let pauseIdList=%str();
%let foundList=%str();
%let RegisteredList=%str();
%let nRegisteredList=0;
%let nFoundList=0;
/* Gotta quote name with macro vars in the next data step */
%let beginQuote='%nrbquote(';
%let endQuote=')';
filename OMABUALL catalog 'work.OMA_Backup.backupall.source';
data _null_;
   drop i p f;
   set OMABUF1.repositories;
   file OMABUALL;
   /* Translate around a problem in macro when using a single paren  */
   /* -- this does not prevent a folder getting created with a paren */
   tname=translate(trim(name),'__','()');
   /* Process only those names passed */
   call symputx("nRegisteredList",trim(_n_));
   %unquote(&subsetIf);
   /* Get the id & name for each repos qualified */
   i+1;
   f=symget('foundList');
   p=symget('pauseIdList');
   if i=1 then
     do;
       call symputx("foundList",trim(name));
       call symputx("pauseIdList",trim(id));
     end;
   else
     do;
        call symputx("foundList",trim(f)||','||trim(name));
        call symputx("pauseIdList",trim(p)||','||trim(id));
     end;
   call symputx("nFoundList",trim(i));
   put '%OMABUREP(' &beginQuote tname &endQuote ','
           id ','
           &beginQuote %unquote("&DestinationPath") &endQuote ','
           &beginQuote %unquote("&ServerStartPath") &endQuote ','
           &beginQuote %unquote("&RposmgrPath") &endQuote ','
           "Reorg=&Reorg" ',' "Restore=&Restore" ');';

run;

/*
%put;
%put DEBUG: addREPOSMGR=&addREPOSMGR;
%put DEBUG: haveAllRepositories=&haveAllRepositories;
%put DEBUG: RepositoryList=&RepositoryList;
%put DEBUG: foundList="&foundList";
%put DEBUG: nRepositoryList=&nRepositoryList;
%put DEBUG: nFoundList=&nFoundList;
%put DEBUG: nRegisteredList=&nRegisteredList;
%put DEBUG: pauseIdList="&pauseIdList";
%put;
*/

%let leaveEarly=0;
/* Error out if we didn't get a validate repository name */
%if %superq(RepositoryList)^=ALL %then
 %do;
   /* Do we have all registered repositories? */
   %if (&nRegisteredList eq &nRepositoryList) AND
        &nRegisteredList eq &nFoundList %then
      %let haveAllRepositories=1;

   /* Adjust for add of REPOSMGR */
   %let temp=%sysfunc(translate(&RepositoryList, %str(  ),%str(%'%")));
   /* Add REPOSMGR if this is a selective RESTORE */
   %if &Restore=N AND &addREPOSMGR eq 1 %then
    %do;
      %let temp=&temp,REPOSMGR;
      %let nRepositoryList = %eval(&nRepositoryList+1);
    %end;

/*
   %put;
   %if &nRegisteredList eq &nRepositoryList AND
       &nRegisteredList eq &nFoundList %then
    %do;
     %put DEBUG: HAVE ALL REGISTERED REPOSITORIES;
    %end;
   %put DEBUG: addREPOSMGR=&addREPOSMGR;
   %put DEBUG: haveAllRepositories=&haveAllRepositories;
   %put DEBUG: RepositoryList="&temp";
   %put DEBUG: foundList="&foundList";
   %put DEBUG: nRepositoryList=&nRepositoryList;
   %put DEBUG: nFoundList=&nFoundList;
   %put DEBUG: nRegisteredList=&nRegisteredList;
   %put DEBUG: pauseIdList="&pauseIdList";
   %put;
*/

   /* If we did not find all the named repositories in the repository list */
   %if &nRepositoryList ne &nFoundList %then
    %do;
     data _null_;
        file print;
        put 'WARNING: Invalid repository list';
        put 'ERROR: No Repositories Will Be Processed';
        put "NOTE: Repository list received: &temp";
        put "NOTE: Repositories registered on the server: &foundList";
        put 'See log for details';
        file log;
      run;
      %put WARNING: Invalid repository list;
      %put ERROR: No Repositories Will Be Processed;
      %put NOTE: Repository List Received: "&temp";
      %put NOTE: Repositories Registered on the server: "&foundList";
      %let leaveEarly=1;
      %goto Leave;
    %end;
 %end;

/*
%put;
%put DEBUG: LEAVEEARLY;
%put;
%let leaveEarly=1;
%goto Leave;
*/

/* Need quotes for parsing purposes. */
%let pauseIdList="&pauseIdList";

/*
%put;
%put DEBUG: pauseIdList=&pauseIdList;
%put;
*/

/* Process Ids into metaoperate option syntax */
/* Process repositories by id's passed to us in the pauseIdList */
/* Find the repository id's delimited by commas */
%let foundREPOSMGR=0;
%if %superq(RepositoryList)^=ALL %then
 %do;
   %let i=1;
   /* Skip quotes */
   %let idList=%qsubstr(&pauseIdList,2,%length(&pauseIdList)-2);
   %do %until (&FQid=%str());
      /* Get the fully qualified id */
      %let FQid=%qscan(&idList,&i,%str(,));
      /* If we have a FQid, get the short id */
      %if &FQid^=%str() %then
         %let shortId=%qsubstr(&FQid,10,8);
      %else
         %let shortId=;
      /* Don't process the REPOSMGR */
      %if  (&shortId=%str(A0000001)) %then
       %do;
         %let foundREPOSMGR=1;
         %goto continue;
       %end;
      %if &shortId^=%str() %then
       %do;
         %if &i eq 1 %then
          %do;
            %let pauseSyntaxList=<REPOSITORY ID="&shortId" STATE="OFFLINE"/>;
          %end;
         %else
          %do;
            %let pauseSyntaxList=&pauseSyntaxList <REPOSITORY ID="&shortId" STATE="OFFLINE"/>;
          %end;
       %end;
      %continue:
      %let i=%eval(&i+1);
   %end;
 %end;
%let pauseSyntaxList=%superq(pauseSyntaxList);
%if &pauseSyntaxList^=%str() %then
   %let options=OPTIONS="&pauseSyntaxList";

/*
%put;
%put DEBUG: addREPOSMGR=&addREPOSMGR;
%put DEBUG: haveAllRepositories=&haveAllRepositories;
%put DEBUG: RepositoryList=&RepositoryList;
%put DEBUG: foundList="&foundList";
%put DEBUG: nRepositoryList=&nRepositoryList;
%put DEBUG: nFoundList=&nFoundList;
%put DEBUG: nRegisteredList=&nRegisteredList;
%put DEBUG: pauseIdList=&pauseIdList;
%put DEBUG: pauseSyntaxList=&pauseSyntaxList;
%put DEBUG: options=&options;
%put;
*/

/*
%put;
%put DEBUG: LEAVEEARLY;
%put;
%let leaveEarly=1;
%goto Leave;
*/

/* Don't pause repositories if only passed the REPOSMGR */
%if &foundREPOSMGR=1 and &pauseSyntaxList=%str() %then
  %let pauseResumeAll=0;
%else
  %let pauseResumeAll=1;

/* Pause all or select list of repositories in the server for OFFLINE */
/* Don't pause repositories if only passed the REPOSMGR */
%if &pauseResumeAll=1 %then
 %do;
   proc metaoperate
     action=pause
     &options;
   run;
 %end;

title Repository States Ready For &TITLETYPE;
%OMABUGRS();
title &TITLETYPE Of Repository;

/* Now do the real work. */
%include OMABUALL;

%Leave:

%if &reorgerr=0 %then
  %do;

   %if &leaveEarly eq 0 %then
    %do;
      /* Resume all repositories in the server */
      /* Don't resume repositories if only passed the REPOSMGR */
      %if &pauseResumeAll=1 %then
       %do;
         proc metaoperate
           action=resume
           &options;
         run;
       %end;
    %end;

   /* Resume the REPOSMGR repository in the server */
   proc metaoperate
        action=resume
        options='<REPOSITORY ID="REPOSMGR"/>';
   run;

   title Final Repository States For &TITLETYPE;
   %OMABUGRS();
   title;

   /* list server status - just because */
   proc metaoperate action=status; run;

   %if &Restore=N %then
    %do;
      %if &RunAnalysis=Y %then
        %do;
         %put NOTE: Running Variable analysis on backup repositories.;
         %put NOTE: DestinationPath=&DestinationPath;
         filename OMAVARAN  catalog 'backdest.OMA_Varan.VarAnalyzeAll.source';
         %include OMAVARAN;
        %end;
      %else
        %do;
         %put NOTE: SAS code to perform variable analysis on backed up repositories has been created.;
          %put NOTE: It can be found in the VarAnalyzeAll.source entry of the OMA_Varan catalog in &testDestPath.;
       %end;
    %end;

   filename OMABUF1;
   filename OMABUF1m;
   libname  OMABUF1 clear;
   filename OMABUF2;
   filename OMABUF2m;
   %if &assignOMABUF2 eq 1 %then
    %do;
      libname OMABUF2 clear;
    %end;
   libname backdest clear; 
  %end;
 %else
  %do;
   %put ;
   %put ERROR: You must resolve Reorg errors before you can resume the metadata server;
  %end;
%getout:
%mend;
